class Cart {
    constructor() {
        const cartElem = document.querySelector('.js-cart');
        this.cartElem = cartElem;
        this.loaderElem = document.querySelector('.loader');
        this.amountInCartElem =  document.querySelector('.amount-in-cart');
        this.totalPriceElem = document.querySelector('.js-total-cart-price');
        this.getItemsUrl = cartElem.dataset.getItemsUrl;
        this.products;
        this.positionsAmount;
        this.totalPrice;
        this.updateItemUrl;
        this.removeItemUrl;
        this.itemUrl;
        this.hasEvents;
        this.updateItemQty = this.updateItemQty.bind(this);
        this.removeItemFromCart = this.removeItemFromCart.bind(this);
    }

    generateProductRowHTML(product) {
        return `<tr>
            <td>
                <div class="row">
                    <div class="col-sm-3 hidden-xs"><img src="/img/${product.item.type}.jpg" width="100" height="100" alt="${product.item.reference}" class="img-responsive"/></div>
                    <div class="col-sm-9">
                        <h4 class="nomargin"><a href="${this.itemUrl + product.item.id}">${product.item.reference}</a></h4>
                        <p>${product.item.description_short}</p>
                    </div>
                </div>
            </td>
            <td>€ ${product.item.price}</td>
            <td>
                <input type="number"
                        min="1"
                        class="js-qty-input form-control text-center"
                        data-product-id="${product.item.id}"
                        value="${product.qty}">
            </td>
            <td class="text-center">€ ${product.subtotal_price}</td>
            <td class="actions text-right">
                <button class="js-remove-btn btn btn-danger btn-sm"
                        data-action="${this.removeItemUrl}"
                        data-product-id="${product.item.id}"
                >Remove</button>
            </td>
        </tr>`;
    }

    showProducts() {
        let productsList = this.products.reduce((acc, item) => acc += this.generateProductRowHTML(item), '');

        this.cartElem.innerHTML = productsList;
        this.amountInCartElem.textContent = this.positionsAmount;
        this.totalPriceElem.textContent = 'Total € ' + this.totalPrice;
    }

    updateItemQty(event) {
        if (!event.target.classList.contains('js-qty-input')) {
            return;
        }
        const qtyInputElem = event.target;
        const changedQty = qtyInputElem.value;
        const actionUrl = this.updateItemUrl;
        const productId = qtyInputElem.dataset.productId;

        if (!changedQty || (changedQty.indexOf('e') + 1) || changedQty === '0') {
            qtyInputElem.value = qtyInputElem.defaultValue;
            return;
        }

        this.loaderElem.classList.toggle('loader-show');

        axios.post(actionUrl, {
            productId,
            changedQty,
        })
            .then( (response) => {
                this.show();
                toastr.success('Quantity updated');
            })
            .catch( (error) => {
                console.log(error);
                this.loaderElem.classList.toggle('loader-show');
                toastr.error('Error, something went wrong');
            });
    }

    removeItemFromCart(event) {
        if (!event.target.classList.contains('js-remove-btn')) {
            return;
        }
        const actionUrl = event.target.dataset.action;
        const productId = event.target.dataset.productId;

        this.loaderElem.classList.toggle('loader-show');

        axios.post(actionUrl, {
            productId,
        })
            .then( (response) => {
                this.show();
                toastr.success('Product deleted');
            })
            .catch( (error) => {
                console.log(error);
                this.loaderElem.classList.toggle('loader-show');
                toastr.error('Error, something went wrong');
            });
    }

    events() {
        this.cartElem.addEventListener('change', this.updateItemQty);
        this.cartElem.addEventListener('click', this.removeItemFromCart);
        this.hasEvents = true;
    }

    show() {
        if (!this.loaderElem.classList.contains('loader-show')) {
            this.loaderElem.classList.toggle('loader-show');
        }

        axios.get(this.getItemsUrl)
            .then( (response) => {
                const redirectLinkWhenEmpty = response.data.redirectLinkWhenEmpty;
                if (redirectLinkWhenEmpty) {
                    window.location.replace(redirectLinkWhenEmpty);
                    return;
                }

                this.products = response.data.products;
                this.positionsAmount = response.data.cart.total_qty;
                this.totalPrice = response.data.cart.total_price;
                this.updateItemUrl = response.data.update_item_url;
                this.removeItemUrl = response.data.remove_item_url;
                this.itemUrl = response.data.item_url;

                this.showProducts();
                this.loaderElem.classList.toggle('loader-show');

                if (!this.hasEvents) {
                    this.events();
                }
            })
            .catch( (error) => {
                console.log(error);
                this.loaderElem.classList.toggle('loader-show');
                toastr.error('Error, something went wrong');
            });
    }
}

const cart = new Cart();
cart.show();
