class Cart {
    constructor() {
        const cartElem = document.querySelector('.js-cart');
        this.cartElem = cartElem;
        this.loaderElem = document.querySelector('.loader');
        this.amountInCartElem =  document.querySelector('.amount-in-cart');
        this.totalPriceElem = document.querySelector('.js-total-cart-price');
        this.getItemsUrl = cartElem.dataset.getItemsUrl;
        this.products;
        this.positionsAmount;
        this.totalPrice;
        this.updateItemUrl = cartElem.dataset.updateItemUrl;
        this.removeItemUrl = cartElem.dataset.removeItemUrl;
        this.itemUrl = cartElem.dataset.itemUrl;
        this.hasEvents;
        this.updateItemQty = this.updateItemQty.bind(this);
        this.removeItemFromCart = this.removeItemFromCart.bind(this);
    }

    generateProductRowHTML(product) {
        return `<tr>
            <td>
                <div class="row">
                    <div class="col-sm-3 hidden-xs"><img src="/img/${product.item.type}.jpg" width="100" height="100" alt="${product.item.reference}" class="img-responsive"/></div>
                    <div class="col-sm-9">
                        <h4 class="nomargin"><a href="${this.itemUrl + product.item.id}">${product.item.reference}</a></h4>
                        <p>${product.item.description_short}</p>
                    </div>
                </div>
            </td>
            <td>€ ${product.item.price}</td>
            <td>
                <input type="number"
                        min="1"
                        class="js-qty-input form-control text-center"
                        data-product-id="${product.item.id}"
                        value="${product.qty}">
            </td>
            <td class="text-center">€ ${product.subtotal_price}</td>
            <td class="actions text-right">
                <button class="js-remove-btn btn btn-danger btn-sm"
                        data-action="${this.removeItemUrl}"
                        data-product-id="${product.item.id}"
                >Remove</button>
            </td>
        </tr>`;
    }

    showProducts() {
        let productsList = this.products.reduce((acc, item) => acc += this.generateProductRowHTML(item), '');

        this.cartElem.innerHTML = productsList;
        this.amountInCartElem.textContent = this.positionsAmount;
        this.totalPriceElem.textContent = 'Total € ' + this.totalPrice;
    }

    toggleLoager() {
        this.loaderElem.classList.toggle('loader-show');
    }

    updateItemQty(event) {
        if (!event.target.classList.contains('js-qty-input')) {
            return;
        }
        const qtyInputElem = event.target;
        const changedQty = qtyInputElem.value;
        const productId = qtyInputElem.dataset.productId;

        if (!changedQty || (changedQty.indexOf('e') + 1) || changedQty === '0') {
            qtyInputElem.value = qtyInputElem.defaultValue;
            return;
        }

        this.toggleLoager();

        this.sendUpdateItemQtyRequest({ productId, changedQty })
            .then( (response) => {
                this.show();
                toastr.success('Quantity updated');
            })
            .catch( (error) => {
                console.log(error);
                this.toggleLoager();
                toastr.error('Error, something went wrong');
            });
    };

    sendUpdateItemQtyRequest(data) {
        return axios.post(this.updateItemUrl, data);
    }

    removeItemFromCart(event) {
        if (!event.target.classList.contains('js-remove-btn')) {
            return;
        }

        const { action, productId } = event.target.dataset;

        this.toggleLoager();

        this.sendRemoveItemRequest({ productId })
            .then( (response) => {
                this.show();
                toastr.success('Product deleted');
            })
            .catch( (error) => {
                console.log(error);
                this.toggleLoager();
                toastr.error('Error, something went wrong');
            });
    }

    sendRemoveItemRequest(data) {
        return axios.post(this.removeItemUrl, data);
    }

    events() {
        this.cartElem.addEventListener('change', this.updateItemQty);
        this.cartElem.addEventListener('click', this.removeItemFromCart);
        this.hasEvents = true;
    }


    show() {
        if (!this.loaderElem.classList.contains('loader-show')) {
            this.loaderElem.classList.toggle('loader-show');
        }

        this.getCurrentProducts()
            .then( (response) => {
                const redirectLink = response.data.redirectLink;
                if (redirectLink) {
                    window.location.replace(redirectLink);
                    return;
                }

                this.products = response.data.products;
                this.positionsAmount = response.data.positionsAmount;
                this.totalPrice = response.data.totalPrice;

                this.showProducts();
                this.toggleLoager();

                if (!this.hasEvents) {
                    this.events();
                }
            })
            .catch( (error) => {
                console.log(error);
                this.toggleLoager();
                toastr.error('Error, something went wrong');
            });
    }

    getCurrentProducts() {
        return axios.get(this.getItemsUrl);
    }
}

const cart = new Cart();
cart.show();
