class Cart {
    constructor() {
        const cartElem = document.querySelector('.js-cart');
        this.cartElem = cartElem;
        this.loaderElem = document.querySelector('.loader');
        this.amountInCartElem =  document.querySelector('.amount-in-cart');
        this.totalPriceElem = document.querySelector('.js-total-cart-price');
        this.getItemsUrl = cartElem.dataset.getItemsUrl;
        this.products;
        this.positionsAmount;
        this.totalPrice;
        this.updateItemUrl = cartElem.dataset.updateItemUrl;
        this.removeItemUrl = cartElem.dataset.removeItemUrl;
        this.itemUrl = cartElem.dataset.itemUrl;
        this.hasEvents;
        this.updateItemQty = this.updateItemQty.bind(this);
        this.removeItemFromCart = this.removeItemFromCart.bind(this);
    }

    generateProductRowHTML(product) {
        return `<tr>
            <td>
                <div class="row">
                    <div class="col-sm-3 hidden-xs"><img src="/img/${product.item.type}.jpg" width="100" height="100" alt="${product.item.reference}" class="img-responsive"/></div>
                    <div class="col-sm-9">
                        <h4 class="nomargin"><a href="${this.itemUrl + product.item.id}">${product.item.reference}</a></h4>
                        <p>${product.item.description_short}</p>
                    </div>
                </div>
            </td>
            <td>€ ${product.item.price}</td>
            <td>
                <input type="number"
                        min="1"
                        class="js-qty-input form-control text-center"
                        data-product-id="${product.item.id}"
                        value="${product.qty}">
            </td>
            <td class="text-center">€ ${product.subtotal_price}</td>
            <td class="actions text-right">
                <button class="js-remove-btn btn btn-danger btn-sm"
                        data-action="${this.removeItemUrl}"
                        data-product-id="${product.item.id}"
                >Remove</button>
            </td>
        </tr>`;
    }

    showProducts() {
        let productsList = this.products.reduce((acc, item) => acc += this.generateProductRowHTML(item), '');

        this.cartElem.innerHTML = productsList;
        this.amountInCartElem.textContent = this.positionsAmount;
        this.totalPriceElem.textContent = 'Total € ' + this.totalPrice;
    }

    toggleLoager() {
        this.loaderElem.classList.toggle('loader-show');
    }

    sendRequest(method, action, data, onsuccess, onerror) {
        let request;
        if (method === 'post') {
            request = axios.post;
        } else if (method === 'get') {
            request = axios.get;
        } else {
            try {
                throw new Error('Expected get or post request');
            } catch (e) {
                console.log(e.name + ': ' + e.message);
            }
        }

        request(action, data)
            .then( (response) => {
                onsuccess(response);
            })
            .catch( (error) => {
                onerror(error);
            });
    }

    updateItemQty(event) {
        if (!event.target.classList.contains('js-qty-input')) {
            return;
        }
        const qtyInputElem = event.target;
        const changedQty = qtyInputElem.value;
        const action = this.updateItemUrl;
        const productId = qtyInputElem.dataset.productId;

        if (!changedQty || (changedQty.indexOf('e') + 1) || changedQty === '0') {
            qtyInputElem.value = qtyInputElem.defaultValue;
            return;
        }

        this.toggleLoager();

        const onsuccess = (response) => {
            this.show();
            toastr.success('Quantity updated');
        };

        const onerror = (error) => {
            console.log(error);
            this.toggleLoager();
            toastr.error('Error, something went wrong');
        }

        this.sendRequest('post', action, { productId, changedQty }, onsuccess, onerror);
    }


    removeItemFromCart(event) {
        if (!event.target.classList.contains('js-remove-btn')) {
            return;
        }

        const { action, productId } = event.target.dataset;

        this.toggleLoager();

        const onsuccess = (response) => {
            this.show();
            toastr.success('Product deleted');
        };

        const onerror = (error) => {
            console.log(error);
            this.toggleLoager();
            toastr.error('Error, something went wrong');
        }

        this.sendRequest('post', action, { productId }, onsuccess, onerror);
    }

    events() {
        this.cartElem.addEventListener('change', this.updateItemQty);
        this.cartElem.addEventListener('click', this.removeItemFromCart);
        this.hasEvents = true;
    }


    show() {
        if (!this.loaderElem.classList.contains('loader-show')) {
            this.loaderElem.classList.toggle('loader-show');
        }

        const onsuccess = (response) => {
            const redirectLink = response.data.redirectLink;
            if (redirectLink) {
                window.location.replace(redirectLink);
                return;
            }

            this.products = response.data.products;
            this.positionsAmount = response.data.positionsAmount;
            this.totalPrice = response.data.totalPrice;

            this.showProducts();
            this.toggleLoager();

            if (!this.hasEvents) {
                this.events();
            }
        };

        const onerror = (error) => {
            console.log(error);
            this.toggleLoager();
            toastr.error('Error, something went wrong');
        }

        this.sendRequest('get', this.getItemsUrl, null, onsuccess, onerror);
    }
}

const cart = new Cart();
cart.show();
