class Cart {
    constructor(cartElem) {
        this.cartHtmlElem = cartElem;
        this.getItemsUrl = cartElem.dataset.getItemsUrl;
    }

    generateProductRow(product) {
        return `<tr>
            <td>
                <div class="row">
                    <div class="col-sm-3 hidden-xs"><img src="/img/${product.item.type}.jpg" width="100" height="100" alt="${product.item.reference}" class="img-responsive"/></div>
                    <div class="col-sm-9">
                        <h4 class="nomargin"><a href="${this.itemUrl + product.item.id}">${product.item.reference}</a></h4>
                        <p>${product.item.description_short}</p>
                    </div>
                </div>
            </td>
            <td>€ ${product.item.price}</td>
            <td>
                <input type="number"
                        min="0"
                        class="js-qty-input form-control text-center"
                        data-product-id="${product.item.id}"
                        value="${product.qty}">
            </td>
            <td class="text-center">€ ${product.subtotal_price}</td>
            <td class="actions text-right">
                <button class="js-remove-btn btn btn-danger btn-sm"
                        data-action="${this.removeItemUrl}"
                        data-product-id="${product.item.id}"
                >Remove</button>
            </td>
        </tr>`;
    }

    showProducts() {
        let productsList = '';

        for (const item in this.products) {
            const product = this.products[item];
            productsList += this.generateProductRow(product);
        }

        this.cartHtmlElem.innerHTML = productsList;
        document.querySelector('.amount-in-cart').textContent = this.positionsAmount;
        document.querySelector('.js-total-cart-price').textContent = 'Total € ' + this.totalPrice;
    }

    setRemoveItemFromCartEventListener() {
        this.bindedRemoveItemFromCart = removeItemFromCart.bind(this);

        document.querySelector('.js-cart').addEventListener('click',  this.bindedRemoveItemFromCart);

        function removeItemFromCart(e) {
            if (!e.target.classList.contains('js-remove-btn')) return;
            const actionUrl = e.target.dataset.action;
            const productId = e.target.dataset.productId;

            axios.post(actionUrl, {
                productId: productId,
            })
                .then( (response) => {
                    const redirectLinkWhenEmpty = response.data.redirectLinkWhenEmpty;
                    if (redirectLinkWhenEmpty) {
                        window.location.replace(redirectLinkWhenEmpty);
                        return;
                    }

                    document.querySelector('.js-cart').removeEventListener('click', this.bindedRemoveItemFromCart);
                    this.show();
                    toastr.success('Product deleted');
                })
                .catch(function (error) {
                    console.log(error);
                    toastr.error('Error, something went wrong');
                });
        }
    }

    setUpdateItemQtyEventListener() {
        document.querySelectorAll('.js-qty-input').forEach(changeQtyInput => {
            let isChanged;
            changeQtyInput.addEventListener('change', (event) => {
                if (isChanged) {
                    return;
                }

                isChanged = true;
                const actionUrl = this.updateItemUrl;
                const productId = event.target.dataset.productId;

                setTimeout(() => {
                    axios.post(actionUrl, {
                        productId: productId,
                        changedQty: event.target.value,
                    })
                        .then( (response) => {
                            document.querySelector('.js-cart').removeEventListener('click', this.bindedRemoveItemFromCart);
                            this.show();
                            toastr.success('Quantity updated');
                        })
                        .catch(function (error) {
                            console.log(error);
                            toastr.error('Error, something went wrong');
                        });

                    isChanged = false;

                }, 1000);
            });
        });
    }

    events() {
        this.setRemoveItemFromCartEventListener();
        this.setUpdateItemQtyEventListener();
    }

    show() {
        axios.get(this.getItemsUrl)
            .then( response => {
                const redirectLinkWhenEmpty = response.data.redirectLinkWhenEmpty;
                if (redirectLinkWhenEmpty) {
                    window.location.replace(redirectLinkWhenEmpty);
                    return;
                }

                this.products = response.data.products;
                this.positionsAmount = response.data.cart.total_qty;
                this.totalPrice = response.data.cart.total_price;
                this.updateItemUrl = response.data.update_item_url;
                this.removeItemUrl = response.data.remove_item_url;
                this.itemUrl = response.data.item_url;

                this.showProducts();
                this.events();
            })
            .catch(function (error) {
                console.log(error);
                alert('Something went wrong');
            });
    }
}

const cartElem = document.querySelector('.js-cart');

const cart = new Cart(cartElem);
cart.show();

