class Cart {
    constructor(cartElem) {
        this.cartHtmlElem = cartElem;
        this.getItemsUrl = cartElem.dataset.getItemsUrl;
    }

    generateProductRow(product) {
        return `<tr>
            <td>
                <div class="row">
                    <div class="col-sm-3 hidden-xs"><img src="/img/${product.item.type}.jpg" width="100" height="100" alt="${product.item.reference}" class="img-responsive"/></div>
                    <div class="col-sm-9">
                        <h4 class="nomargin"><a href="${this.itemUrl + product.item.id}">${product.item.reference}</a></h4>
                        <p>${product.item.description_short}</p>
                    </div>
                </div>
            </td>
            <td>€ ${product.item.price}</td>
            <td>
                <input type="number"
                        min="1"
                        class="js-qty-input form-control text-center"
                        data-product-id="${product.item.id}"
                        value="${product.qty}">
            </td>
            <td class="text-center">€ ${product.subtotal_price}</td>
            <td class="actions text-right">
                <button class="js-remove-btn btn btn-danger btn-sm"
                        data-action="${this.removeItemUrl}"
                        data-product-id="${product.item.id}"
                >Remove</button>
            </td>
        </tr>`;
    }

    showProducts() {
        let productsList = '';

        for (const item in this.products) {
            const product = this.products[item];
            productsList += this.generateProductRow(product);
        }

        this.cartHtmlElem.innerHTML = productsList;
        document.querySelector('.amount-in-cart').textContent = this.positionsAmount;
        document.querySelector('.js-total-cart-price').textContent = 'Total € ' + this.totalPrice;
    }

    updateItemQty(event) {
        if (!event.target.classList.contains('js-qty-input')) {
            return;
        }

        const changedQty = event.target.value;
        if (!changedQty || (changedQty.indexOf('e') + 1) || changedQty === '0') {
            event.target.value = event.target.defaultValue;
            return;
        }

        const actionUrl = this.updateItemUrl;
        const productId = event.target.dataset.productId;

        loaderElem.classList.toggle('loader-show');

        axios.post(actionUrl, {
            productId: productId,
            changedQty: changedQty,
        })
            .then( (response) => {
                this.show();
                toastr.success('Quantity updated');
            })
            .catch(function (error) {
                console.log(error);
                loaderElem.classList.toggle('loader-show');
                toastr.error('Error, something went wrong');
            });
    }

    removeItemFromCart(event) {
        if (!event.target.classList.contains('js-remove-btn')) {
            return;
        }
        const actionUrl = event.target.dataset.action;
        const productId = event.target.dataset.productId;

        loaderElem.classList.toggle('loader-show');

        axios.post(actionUrl, {
            productId: productId,
        })
            .then( (response) => {
                const redirectLinkWhenEmpty = response.data.redirectLinkWhenEmpty;
                if (redirectLinkWhenEmpty) {
                    window.location.replace(redirectLinkWhenEmpty);
                    return;
                }
                this.show();
                toastr.success('Product deleted');
            })
            .catch(function (error) {
                console.log(error);
                loaderElem.classList.toggle('loader-show');
                toastr.error('Error, something went wrong');
            });
    }

    events() {
        this.cartHtmlElem.addEventListener('change', this.updateItemQty.bind(this));
        this.cartHtmlElem.addEventListener('click', this.removeItemFromCart.bind(this));
        this.hasEvents = true;
    }

    show() {
        if (!loaderElem.classList.contains('loader-show')) {
            loaderElem.classList.toggle('loader-show');
        }

        axios.get(this.getItemsUrl)
            .then( response => {
                const redirectLinkWhenEmpty = response.data.redirectLinkWhenEmpty;
                if (redirectLinkWhenEmpty) {
                    window.location.replace(redirectLinkWhenEmpty);
                    return;
                }

                this.products = response.data.products;
                this.positionsAmount = response.data.cart.total_qty;
                this.totalPrice = response.data.cart.total_price;
                this.updateItemUrl = response.data.update_item_url;
                this.removeItemUrl = response.data.remove_item_url;
                this.itemUrl = response.data.item_url;

                this.showProducts();
                loaderElem.classList.toggle('loader-show');

                if (!this.hasEvents) {
                    this.events();
                }
            })
            .catch(function (error) {
                console.log(error);
                loaderElem.classList.toggle('loader-show');
                toastr.error('Error, something went wrong');
            });
    }
}

const loaderElem = document.querySelector('.loader');

const cartElem = document.querySelector('.js-cart');
const cart = new Cart(cartElem);
cart.show();
